<?php

// Mall functions
function getMalls($pdo) {
    $stmt = $pdo->query("SELECT * FROM malls WHERE status = 'active' ORDER BY display_order");
    return $stmt->fetchAll();
}

function getMallBySlug($pdo, $slug) {
    $stmt = $pdo->prepare("SELECT * FROM malls WHERE slug = ? AND status = 'active'");
    $stmt->execute([$slug]);
    return $stmt->fetch();
}

// Movie functions
function getFeaturedMovies($pdo, $limit = 3) {
    $stmt = $pdo->prepare("SELECT * FROM movies WHERE status = 'now_showing' ORDER BY created_at DESC LIMIT ?");
    $stmt->execute([$limit]);
    return $stmt->fetchAll();
}

function getMovieById($pdo, $id) {
    $stmt = $pdo->prepare("SELECT * FROM movies WHERE id = ?");
    $stmt->execute([$id]);
    return $stmt->fetch();
}

// News/Events functions
function getRecentNews($pdo, $limit = 3) {
    $stmt = $pdo->prepare("
        SELECT n.*, m.name as mall_name
        FROM news_events n
        LEFT JOIN malls m ON n.mall_id = m.id
        WHERE n.status = 'published'
        ORDER BY n.created_at DESC
        LIMIT ?
    ");
    $stmt->execute([$limit]);
    return $stmt->fetchAll();
}

function getNewsById($pdo, $id) {
    $stmt = $pdo->prepare("
        SELECT n.*, m.name as mall_name
        FROM news_events n
        LEFT JOIN malls m ON n.mall_id = m.id
        WHERE n.id = ?
    ");
    $stmt->execute([$id]);
    return $stmt->fetch();
}

function getNewsByType($pdo, $type, $limit = null) {
    $sql = "
        SELECT n.*, m.name as mall_name
        FROM news_events n
        LEFT JOIN malls m ON n.mall_id = m.id
        WHERE n.type = ? AND n.status = 'published'
        ORDER BY n.created_at DESC
    ";

    if ($limit) {
        $sql .= " LIMIT ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$type, $limit]);
    } else {
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$type]);
    }

    return $stmt->fetchAll();
}

// Store functions
function getFeaturedStores($pdo, $limit = 4) {
    $stmt = $pdo->prepare("
        SELECT s.*, m.name as mall_name,
               CONCAT(s.floor, ', ', m.name) as location
        FROM stores s
        LEFT JOIN malls m ON s.mall_id = m.id
        WHERE s.status = 'active' AND s.featured = 1
        ORDER BY s.created_at DESC
        LIMIT ?
    ");
    $stmt->execute([$limit]);
    return $stmt->fetchAll();
}

function getStoresByMall($pdo, $mall_id) {
    $stmt = $pdo->prepare("SELECT * FROM stores WHERE mall_id = ? AND status = 'active' ORDER BY name");
    $stmt->execute([$mall_id]);
    return $stmt->fetchAll();
}

function getStoresByCategory($pdo, $category, $mall_id = null) {
    if ($mall_id) {
        $stmt = $pdo->prepare("SELECT * FROM stores WHERE category = ? AND mall_id = ? AND status = 'active' ORDER BY name");
        $stmt->execute([$category, $mall_id]);
    } else {
        $stmt = $pdo->prepare("SELECT * FROM stores WHERE category = ? AND status = 'active' ORDER BY name");
        $stmt->execute([$category]);
    }
    return $stmt->fetchAll();
}

// Community Card functions
function createCommunityCard($pdo, $data) {
    $stmt = $pdo->prepare("
        INSERT INTO community_cards (user_name, email, phone, card_number, status, created_at)
        VALUES (?, ?, ?, ?, 'active', NOW())
    ");

    $card_number = generateCardNumber();
    return $stmt->execute([
        $data['user_name'],
        $data['email'],
        $data['phone'],
        $card_number
    ]);
}

function getCommunityCardByEmail($pdo, $email) {
    $stmt = $pdo->prepare("SELECT * FROM community_cards WHERE email = ?");
    $stmt->execute([$email]);
    return $stmt->fetch();
}

function generateCardNumber() {
    return 'OCC' . date('Y') . str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
}

// Newsletter functions
function subscribeNewsletter($pdo, $email) {
    // Check if email already exists
    $stmt = $pdo->prepare("SELECT id FROM newsletter_subscriptions WHERE email = ?");
    $stmt->execute([$email]);

    if ($stmt->fetch()) {
        return false; // Already subscribed
    }

    $stmt = $pdo->prepare("INSERT INTO newsletter_subscriptions (email, subscribed_at) VALUES (?, NOW())");
    return $stmt->execute([$email]);
}

// Search functions
function searchStores($pdo, $query, $mall_id = null) {
    $searchTerm = "%$query%";

    if ($mall_id) {
        $stmt = $pdo->prepare("
            SELECT s.*, m.name as mall_name
            FROM stores s
            LEFT JOIN malls m ON s.mall_id = m.id
            WHERE (s.name LIKE ? OR s.description LIKE ?)
            AND s.mall_id = ? AND s.status = 'active'
            ORDER BY s.name
        ");
        $stmt->execute([$searchTerm, $searchTerm, $mall_id]);
    } else {
        $stmt = $pdo->prepare("
            SELECT s.*, m.name as mall_name
            FROM stores s
            LEFT JOIN malls m ON s.mall_id = m.id
            WHERE (s.name LIKE ? OR s.description LIKE ?)
            AND s.status = 'active'
            ORDER BY s.name
        ");
        $stmt->execute([$searchTerm, $searchTerm]);
    }

    return $stmt->fetchAll();
}

function searchMovies($pdo, $query) {
    $searchTerm = "%$query%";
    $stmt = $pdo->prepare("
        SELECT * FROM movies
        WHERE (title LIKE ? OR description LIKE ?)
        AND status = 'now_showing'
        ORDER BY title
    ");
    $stmt->execute([$searchTerm, $searchTerm]);
    return $stmt->fetchAll();
}

function searchNews($pdo, $query) {
    $searchTerm = "%$query%";
    $stmt = $pdo->prepare("
        SELECT n.*, m.name as mall_name
        FROM news_events n
        LEFT JOIN malls m ON n.mall_id = m.id
        WHERE (n.title LIKE ? OR n.content LIKE ?)
        AND n.status = 'published'
        ORDER BY n.created_at DESC
    ");
    $stmt->execute([$searchTerm, $searchTerm]);
    return $stmt->fetchAll();
}

// Utility functions
function formatDate($date) {
    return date('F j, Y', strtotime($date));
}

function formatDateTime($datetime) {
    return date('F j, Y \a\t g:i A', strtotime($datetime));
}

function truncateText($text, $length = 150) {
    if (strlen($text) <= $length) {
        return $text;
    }
    return substr($text, 0, $length) . '...';
}

function sanitizeInput($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

// Admin functions
function isAdmin() {
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

function requireAdmin() {
    if (!isAdmin()) {
        header('Location: admin/login.php');
        exit;
    }
}
?>
